"use strict";
/***********************************************************/
/*  BitSight Archer Integration - Get BAA Vector Ratings   */
/***********************************************************/

/*
Purpose: 
	Gets new requests for BitSight BAA/Vectors Ratings (score), use a one-time license, and updates the Portfolio record(s)
	NOTE: This can ONLY be done via API

High Level Overview:
	Obtain BitSight Portfolios who wish to pull BAA/Vector Ratings
		Criteria: [Pull Todays BitSight Risk Vectors?] is "Yes"
	Use the BitSite API to use a license and obtain the detailed vector ratings
	Reset the BitSight Portfolio flag and update the vectors in Archer

Detailed Process: 
	1. Logs into Archer to get session token
	2. Obtain the Archer FieldIDs for the content post via api from the app
	3. Obtain Archer values list IDs programmatically via api
	4. Obtain list of Archer BitSight Portfolio records which need the BAA/Vectors
	5. Retrive Customer Name from BitSight to pass into x-BitSight-Customer header param
	6. Uses BitSight API to use a BAA license token and captures vector results per portfolio
	7. Builds postbody for each Portfolio record with vector rating info
	8. Attempts to update Portfolio record(s) in Archer with vector rating data and reseting the flag for obtaining the license
*/

/**********************************************************/
/* VERSIONING                                             */
/**********************************************************/
/*  
	1/12/2022 - Version 2.0
    Initial Version - 
*/


/********************************************************/
/********	ARCHER DFM APPROVED LIBRARIES
/********************************************************/
var request = require('request');
//var xpath = require('xpath');
var xmldom = require('xmldom');
var xml2js = require('xml2js');

/********************************************************/
/***** DEFAULT PARAMS
/********************************************************/
/* These params will be used unless overriden by parameters passed in. */
var defaultParams = {
	'archer_username':"dfm_BitSightAPIDaily",				//Example used in Installation Guide. Update name if you change it.
	'archer_password':"YOURPASSWORD",
	'archer_instanceName':'INSTANCENAME', 					//Example: 88555 or Dev									Instance name is case-sensitive. SaaS instances are typically numbers
	'archer_webroot':"https://ARCHERURL.com/",			//Example: https://[ArcherDomainURL]/		Must include trailing slash
	'archer_webpath':"RSAarcher/", 									//Example: RSAarcher/										Must include trailing slash UNLESS there is no virtual directory...then leave blank - Preconfigured for Archer SaaS environment
	'archer_ws_root':"RSAarcher/ws/",								//Example: RSAarcher/ws/								Must include trailing slash and vdir is case-sensitive - Preconfigured for Archer SaaS environment
	'archer_rest_root':"RSAArcher/platformapi/",		//Example: RSAarcher/platformapi/				Must include trailing slash and vdir is case-sensitive - Preconfigured for Archer SaaS environment

	'bitsight_token':"YOUR_BITSIGHT_TOKEN",     		//BitSight API Token for authentication


	//Web call options
	'proxy':'',
	'verifyCerts':'true',

	//Application Names
	'archer_ThirdPartyProfileApp':'Third Party Profile', 				//Name of the "Third Party Profile" application in Archer for your organization. 
	'archer_BitSightPortfolioApp':'BitSight Portfolio', 				//Name of the "BitSight Portfolio" ODA in Archer for your organization. 

	//Typically these will not change
	'bitsight_webroot':"https://api.bitsighttech.com/",     			//Example: https://api.bitsighttech.com/		Main BitSight API URL								*Must include trailing slash    
    'bitsight_getbaaratings':"ratings/v1/companies/",    				//Example: ratings/v1/companies/				Endpoint for getting BAA ratings					*Must include trailing slash    
	'bitsight_currentuser':"users/current",       						//Example: users/current 						Endpoint for getting current user/company info
	'bitsight_weblink':"https://service.bitsighttech.com/app/spm/company/", //Example: https://service.bitsighttech.com/app/spm/company/	URL before the Company GUID		*Must include trailing slash
    'bitsight_baa_assessment':"b69d6a51-f500-4fcf-ad98-8fd90ea2adc4", //Example: b69d6a51-f500-4fcf-ad98-8fd90ea2adc4	BitSight BAA Assessment ID to pull vectors

	//Typically these will not change unless Archer changes the structure	
	'archer_loginpath':"general.asmx",									//Example: general.asmx
    'archer_searchpath': "search.asmx",									//Example: search.asmx
	'archer_contentpath': "core/content",								//Example: core/content				//Do NOT include trailing slash
	'archer_applicationpath': "core/system/application/",							//Example: core/system/application/							Must include trailing slash
	'archer_fielddefinitionapppath': "core/system/fielddefinition/application/",	//Example: core/system/fielddefinition/application/			Must include trailing slash
	'archer_fielddefinitionpath': "core/system/fielddefinition/",					//Example: core/system/fielddefinition/						Must include trailing slash
	'archer_valueslistvaluepath':"core/system/valueslistvalue/flat/valueslist/",	//Example: core/system/valueslistvalue/flat/valueslist/		Must include trailing slash
	'archer_version':"core/system/applicationinfo/version",							//Example: core/system/applicationinfo/version				Do NOT include trailing slash

	
	//*****************DO NOT MODIFY************************
	'bIsSaaS':"true", //**Must be set to true! In the future, this may be converted to a JST data feed
	'bDFDebugging':"false" //**Must be set to false !Only applicable for Data Feed Jobs. Always set to false for SaaS.
};


var bVerboseLogging = false; //Verbose logging will log the post body data and create output files which takes up more disk space. Very useful for troubleshooting, but not necessary unless you are having issues.
var bIsLegacyIntegration = false; //Set this to true if you are using the Archer 6.7 version of the package. If you are using Archer 6.10 P1 or higher, set this to: false

var SaaSParams =  {}; //This is automatically populated with all the Archer field IDs and values list IDs making deployment between environments easy.
var appPrefix = '130'; //Used to store files in logs subdirectory when VerboseLogging is enabled.

/********************************************************/
/********	DEBUGGING/TESTING SETTINGS
/********************************************************/
var testingMode_Tokens = {
	'LastRunTime':'2018-06-13T18:31:41Z',
	'PreviousRunContext':''
};

var testingMode_Params = {};

/********************************************************/
/***** GLOBAL VARS
/********************************************************/
var testingMode = true;
var sSessionToken = null;
var params = null;
var tokens = null;
var verifyCerts = false;
var useProxy = false;
var errorArray = [];
var previousRunContext = {};
var aBitSightRatings = []; //Final object returned to Archer data feed
var ArcherValuesLists = []; //This will be used for iterating through to obtain values list value id from Archer
var bSaaSErrorFound = false;
var sDebug = '';
var iCurrentCompany = 0;
var sArcherVersionNum = "TBD";

//Just a simple way to remember the Archer input data types
var ArcherInputTypes = 
{
    'text':1,
    'numeric':2,
    'date':3,
    'valueslist':4,
    'externallinks':7,
    'usergrouprecperm':8,
    'crossref':9,
    'attachment':11,
    'image':12,
    'matrix':16,
    'ipaddress':19,
    'relatedrecords':23,
    'subform':24
}

var COST_ArcherBitSightVersion = "RSA Archer 1.0";
var COST_Platform = "RSA Archer";
var BitSightCustomerName = "unknown";

/********************************************************/
/********	INIT
/********************************************************/
function init(){
	/* run the feed */
	//LogInfo("Datafeed Init");
	/* check if testing mode should be active (no archer DFE present) */
	if(typeof context == 'undefined'){
		testingMode=true;
	}
	/* get params and tokens */
	params = getArcherParams();
	tokens = getArcherTokens();
    /* setup cert verify */
    if(params['verifyCerts'].toLowerCase()=='true'){
        verifyCerts=true;
	}else{
		process.env.NODE_TLS_REJECT_UNAUTHORIZED = "0";
	}
	/* setup proxy */
	if(params['proxy']!=null && params['proxy']!=''){
		useProxy=true;
	}
	/* get the last run time */
	var lrt = new Date('1970-01-10T00:00:00Z');
	/* check for last run time */
	if('LastRunTime' in tokens && tokens['LastRunTime']!=null && tokens['LastRunTime']!=''){
		lrt = new Date(tokens['LastRunTime']);
	}
	//LogInfo("Last Datafeed Run Time: " + lrt.toISOString());
	return lrt;
}

/********************************************************/
/********	DATA CONVERSIONS
/********************************************************/
function jsonToXMLString(json,rootElement){
	if(rootElement){
		var bldrOpts = {
			headless: true, rootName: rootElement, renderOpts: {'pretty': true,'indent': '    ','newline': '\r\n','cdata': true}
		}
	}else{
		var bldrOpts = {
			headless: true, renderOpts: {'pretty': true,'indent': '    ','newline': '\r\n','cdata': true}
		}
	}
	return new xml2js.Builder(bldrOpts).buildObject(json);
}

function jsonToString(json){
	return JSON.stringify(json,null,4);
}

function xmlStringToJSON(xml,callBack){
	xml2js.parseString(xml, {}, callBack);
}

function xmlStringToXmlDoc(xml){
	var p  = new xmldom.DOMParser;
	return p.parseFromString(xml);
}

function jsonArrayToXMLBuffer(jsonArray,elementName){
	/* holds the buffers */
	var buffers = [];
	/* conver each element to an xml buffer */
	for(var i in jsonArray){
		/* convert it to xml */
		var xmlString = jsonToXMLString(jsonArray[i],elementName);
		/* convert it to a buffer */
		var b = Buffer.from(xmlString+'\n', 'utf8');
		/* add to buffers array */
		buffers.push(b);
	}
	/* concat to one giant buffer */
	return Buffer.concat(buffers);
}

/********************************************************/
/********	ARCHER CALLBACK INTERFACE
/********************************************************/
function getArcherParams(){
	var params = null;
	/* which params should we use? */
	if(testingMode){
		params=testingMode_Params;
	}else{
		params=context.CustomParameters;
	}
	/* now add the default params */
	for(var k in defaultParams){
		if(!(k in params)){
			params[k]=defaultParams[k];
		}
	}
	/* return them */
	return params;
}

function getArcherTokens(){
	if(testingMode){
		return testingMode_Tokens;
	}
	return context.Tokens;
}

var dfCallback = function(err,data){
	if(err){
		LogError("ERROR: Datafeed Failure due to error.");
		if (params['bIsSaaS'] == "true"){
			LogSaaSError();
		}else{
			if (params['bDFDebugging'] == "true" || params['bDFDebugging'] == true)
			{
				LogInfo("Data Feed Debugging Enabled");
				//The commented line sends the data to a record in Archer, but the record ID needs to be provided.
				var start = Buffer.from('<Records><Record><![CDATA[\n', 'utf8');
				var middle = Buffer.from(sDebug,'utf8');
				//var end = Buffer.from(']]></Report_Results_HTML></Record></Records>\n', 'utf8');
				var end = Buffer.from(']]></Record></Records>\n', 'utf8');
				var returntoArcher = Buffer.concat([start,middle,end]);
				var sXMLReturn = returntoArcher.toString('utf8');
				sendDataToArcher(sXMLReturn);
			}
			else{
				sendDataToArcher(null);
			}
		}
	}else{
		if (params['bIsSaaS'] == "true"){
			LogSaaSSuccess();
			LogInfo("SUCCESS: Overall Success");
		}else{		
			LogInfo("Exporting Data to Archer.");
			sendDataToArcher(data);
		}
	}
};

function sendDataToArcher(data)
{
	if(testingMode || bVerboseLogging){
		if (data){
			var fs = require('fs'); 
			fs.writeFile("logs\\" + appPrefix + "\\" + "!ReturnedToArcher.xml",data,function(err)
			{
				if(err) 
				{
					LogError("ERROR SAVING FILE IN TEST MODE: " + err);
				}
			});
		}
		if(errorArray.length>0){
			for(var i in errorArray){
				LogError(errorArray[i]);
			}
		}
	}
	else{
		/* check for any errors */
		if (errorArray.length>0){
			//callback(errorArray,{output: data, previousRunContext: JSON.stringify(previousRunContext)}); //Only if data came back as JSON
			callback(errorArray,{output: data, previousRunContext: JSON.stringify(previousRunContext)});  //Attempting to use the XML returned from the Archer API
		}else{
			//callback(null,{output: data, previousRunContext: JSON.stringify(previousRunContext)});  //Only if data came back as JSON
			callback(null,{output: data, previousRunContext: JSON.stringify(previousRunContext)}); //Attempting to use the XML returned from the Archer API
		}
	}
}

/********************************************************/
/********	ERROR HANDLING AND LOGGING
/********************************************************/
function captureError(text){
	LogSaaSError();

	// if(text!=null)
	// {
	// 	/* create a new error to get the stack */
	// 	var e = new Error();
	// 	/* create error string for array */
	// 	var errString = text + "\n" + e.stack;
	// 	/* add to error array */
	// 	errorArray.push(errString);
	// }

}

function getDateTime(){
    var dt = new Date();
    return pad(dt.getFullYear(),4)+"-"+pad(dt.getMonth()+1,2)+"-"+pad(dt.getDate(),2)+" "+pad(dt.getHours(),2)+":"+pad(dt.getMinutes(),2)+":"+pad(dt.getSeconds(),2);
}

function LogInfo(text){
    console.log(getDateTime() + " :: INFO  :: " + text);
	if (params['bDFDebugging'] == "true" || params['bDFDebugging'] == true)
	{
		appendDebug(text);
	}
}

function LogError(text){
	LogSaaSError();
	console.log(getDateTime() + " :: ERROR :: " + text);
	if (params['bDFDebugging'] == "true" || params['bDFDebugging'] == true)
	{
		appendDebug(text);
	}
}

function LogWarn(text){
    console.log(getDateTime() + " :: WARN  :: " + text);
	if (params['bDFDebugging'] == "true" || params['bDFDebugging'] == true)
	{
		appendDebug(text);
	}
}

function LogVerbose(text){
	if (bVerboseLogging){
		LogInfo(text);
	}
}

function pad(num, size){
    var s = num+"";
    while (s.length < size) s = "0" + s;
    return s;
}

/********************************************************/
/********	LogSaaSError
/********************************************************/
var LogSaaSError = function()
{
	//Simple method to log an error to a file for SaaS for batch execution. The batch file will check if this file exists.
	if (params['bIsSaaS'] == "true" && bSaaSErrorFound == false){
		bSaaSErrorFound = true;  //Only log if we haven't found an error to help avoid file lock issues.
		var fs = require('fs'); 			
		fs.writeFileSync("logs\\error" + "-" + appPrefix + ".txt","ERROR");
		LogInfo("Logged error and created logs\\error" + "-" + appPrefix + ".txt file.");
	}
}

/************************************************************************************************************************************************************************************************************************************/
/********	LogSaaSSuccess
/************************************************************************************************************************************************************************************************************************************/
var LogSaaSSuccess = function()
{
	//Simple method to log successful execution for SaaS for batch execution. The batch file will check if this file exists.
	if (params['bIsSaaS'] == "true" && bSaaSErrorFound == false){
		var fs = require('fs'); 			
		fs.writeFileSync("logs\\success" + "-" + appPrefix + ".txt","SUCCESS");
		LogInfo("Logged success and created logs\\success" + "-" + appPrefix + ".txt file.");
	}
}



//GetArcherText validates then returns data from a text value
function GetArcherText(sText)
{
	if (typeof sText == 'undefined' || 
		typeof sText._ == 'undefined' || 
		sText == null)
	{
		return "";
	}else{
		return sText._;
	}
}


//GetArcherValue validates then returns data from a single value list
function GetArcherValue(jValueNode)
{
	if (typeof jValueNode == 'undefined' || 
		jValueNode == null ||
		typeof jValueNode.ListValues == 'undefined' || 
		typeof jValueNode.ListValues[0] == 'undefined' || 
		typeof jValueNode.ListValues[0].ListValue == 'undefined' || 
		typeof jValueNode.ListValues[0].ListValue[0] == 'undefined' || 
		typeof jValueNode.ListValues[0].ListValue[0]._ == 'undefined')
	{
		return "";
	}else{
		return jValueNode.ListValues[0].ListValue[0]._;
	}
}




/********************************************************/
/********	WEB CALL
/********************************************************/
function webCall(url,method,headers,postBody,callBack)
{
	/* build options */
	var options = {
		method: method,
		uri: url,
		headers: headers,
		body: postBody,
		rejectUnauthorized: verifyCerts
	}
	/* add in proxy */
	if(useProxy)
	{
		options['proxy']=params['proxy'];
	}
	
	//LogInfo("HERE url: " + url);
	//LogInfo("HERE method: " + method);
	//LogInfo("HERE postBody: " + postBody);
	
	/* make the request */
	request(options,
		function handleResponse(err, response, body)
		{
			var errMessage = null;
			var headers = {};				
			/* check for error */
			if(err)
			{
				errMessage='HTTP ERROR: ' + err;
			}
			else
			{
				headers = response.headers;
				//LogInfo("HERE headers: " + headers);
				if(response.statusCode!=200)
				{
					errMessage='HTTP ' + response.statusCode + ' ERROR: ' + err;
				}
			}
			/* check for error */
			if(errMessage)
			{
				captureError('Error in HTTP Response: ' + errMessage);
				callBack(true,headers,body);
			}
			else
			{
				callBack(false,headers,body);
			}
		}
	);
}

function appendDebug(sTxt)
{
	sDebug+=sTxt.toString('utf8') + "\n";
	//sDebug+=sTxt + "|";
}

function b64Encode(str) {  
    var CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";  
    var out = "", i = 0, len = str.length, c1, c2, c3;  

     while (i < len) {  
                      c1 = str.charCodeAt(i++) & 0xff;  
                      if (i == len) {  
                                       out += CHARS.charAt(c1 >> 2);  
                                       out += CHARS.charAt((c1 & 0x3) << 4);  
                                       out += "==";  
                                       break;  
     }  

     c2 = str.charCodeAt(i++);  
     if (i == len) {  
                      out += CHARS.charAt(c1 >> 2);  
                      out += CHARS.charAt(((c1 & 0x3)<< 4) | ((c2 & 0xF0) >> 4));  
                      out += CHARS.charAt((c2 & 0xF) << 2);  
                      out += "=";  
                      break;  
     }  

     c3 = str.charCodeAt(i++);  
     out += CHARS.charAt(c1 >> 2);  
     out += CHARS.charAt(((c1 & 0x3) << 4) | ((c2 & 0xF0) >>4));  
     out += CHARS.charAt(((c2 & 0xF) << 2) | ((c3 & 0xC0) >>6));  
     out += CHARS.charAt(c3 & 0x3F);  
    }  
     return out;  
} 

function makeBasicAuth(token)
{
    //Purpose of this is to convert the token to the authorization header for basic auth
    //Format is Token with a colon at the end then converted to Base64
    return b64Encode(token + ":");
}


/********************************************************/
/********	ArcherAPI_Login
/********************************************************/
function ArcherAPI_Login(callBack)
{
	appendDebug("Login");
	/* build url */
	var url =  params['archer_webroot'] + params['archer_ws_root'] + params['archer_loginpath'];

	var postBody = "<?xml version=\"1.0\" encoding=\"utf-8\"?><soap:Envelope xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\">" +
				   "<soap:Body><CreateUserSessionFromInstance xmlns=\"http://archer-tech.com/webservices/\"><userName>" + params['archer_username'] + "</userName><instanceName>" + params['archer_instanceName'] + "</instanceName><password>" + params['archer_password'] + "</password></CreateUserSessionFromInstance></soap:Body></soap:Envelope>";
	
	var headers = {
			'content-type' : 'text/xml; charset=utf-8'
	}
	
	var results = []; 

	var loginToArcher = function(url){
		LogInfo('----------------------------Login Attempt Starting------------------');
		LogInfo('URL1: ' + url);
		LogVerbose('POST1: ' + postBody);
		//LogVerbose('HEAD: ' + headers);

		webCall(url,'POST',headers,postBody,function(err,h,b)
		{
			if(err)
			{
				LogError("ERR1: " + err);
				//LogVerbose("HEAD1: " + h);
				LogError("RESP1: " + b);
				appendDebug("ERR1");
				callBack(true,null);
			}
			else
			{
				//LogInfo("webCall Results: " + b);
				//This is the result from the webCall
				var doc = xmlStringToXmlDoc(b);
				//Get the value of the the text which is the session token
				sSessionToken = doc.getElementsByTagName("CreateUserSessionFromInstanceResult")[0].childNodes[0].nodeValue;
				LogInfo("sSessionToken: " + sSessionToken);
				if (sSessionToken.length > 5)
				{
					GetArcherVersion(callBack);
				}
				else
				{
					LogError("sSessionToken blank: " + sSessionToken);
					appendDebug("SessionBlank");
					callBack(true,null);
				}
			}
		});
	}

	/* kick off process */
	loginToArcher(url);
}

/********************************************************/
/********	GetArcherVersion
/********************************************************/
var GetArcherVersion = function(callBack)
{
	LogInfo('----------------------------GetArcherVersion----------------------------');

	var url =  params['archer_webroot'] + params['archer_rest_root'] + params['archer_version'];

	var headers = {	'Authorization' : 'Archer session-id="' + sSessionToken + '"',
					'Content-type': 'application/json',
					'Accept': 'application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8'};
	
	var postBody = {};

	LogInfo('URL1.1: ' + url);
	LogVerbose('BODY1.1: ' + jsonToString(postBody));
	//LogInfo('HEAD: ' + headers);

	webCall(url,'GET',headers,jsonToString(postBody),function(err,h,b)
	{
		if(err)
		{
			LogSaaSError();
			LogError("ERR1.1: " + err);
			//LogInfo("HERE HEAD1: " + h);
			LogError("BODY1.1: " + b);
		}
		else
		{
			LogVerbose("Raw Report Results: " + b);
			var resultJSON = JSON.parse(b);

			if(typeof resultJSON != 'undefined' && typeof resultJSON.RequestedObject != 'undefined' && resultJSON.RequestedObject.Version != 'undefined')
			{
				sArcherVersionNum = resultJSON.RequestedObject.Version; //Get the Version
				COST_Platform = COST_Platform + " (" + sArcherVersionNum + ")";
				LogInfo("sArcherVersionNum: " + sArcherVersionNum);
				getPortfolioFieldIDs(callBack);
			}
			else
			{
				sArcherVersionNum = "Unknown";
				LogWarn("ERROR Obtaining Archer Version Number: " + b);
				getPortfolioFieldIDs(callBack);
			}
		}
	});
}



/********************************************************/
/********	getPortfolioFieldIDs
/********************************************************/
var getPortfolioFieldIDs = function(callBack)
{
	LogInfo('----------------------------getPortfolioFieldIDs----------------------------');
	var getPortfolioModuleID = function(callBack)
	{
		LogInfo('----------------------------getPortfolioAppID----------------------------');
		var url =  params['archer_webroot'] + params['archer_rest_root'] + params['archer_applicationpath'];

		var headers = {	'Authorization' : 'Archer session-id="' + sSessionToken + '"',
						'Content-type': 'application/json',
						'Accept': 'application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8'};
		
		var odataquery = "?$select=Name,Id,Guid&$filter=Name eq '" + params['archer_BitSightPortfolioApp'] + "'";
		var postBody = {
			"Value": odataquery
		};	

		LogInfo('URL2: ' + url);
		LogVerbose('BODY2: ' + jsonToString(postBody));
		//LogVerbose('HEAD: ' + headers);

		webCall(url,'GET',headers,jsonToString(postBody),function(err,h,b)
		{
			if(err)
			{
				LogError("ERR2: " + err);
				//LogVerbose("HERE HEAD1: " + h);
				LogError("BODY2: " + b);
			}
			else
			{
				LogVerbose("Raw Results: " + b);
				var resultJSON = JSON.parse(b);

				if(typeof resultJSON != 'undefined' && typeof resultJSON[0].RequestedObject != 'undefined' && resultJSON[0].RequestedObject.Id != 'undefined')
				{
					var iModuleID = resultJSON[0].RequestedObject.Id; //Get the content ID
					LogInfo("iModuleID: " + iModuleID);
					getFieldIDs(iModuleID,callBack);
				}
				else
				{
					LogError("ERROR Obtaining BitSight Portfolio module ID: " + b);
				}
			}
		});
	}

	var getFieldIDs = function(iModuleID,callBack)
	{
		LogInfo('----------------------------getPortfolioFieldIDs----------------------------');
		var url =  params['archer_webroot'] + params['archer_rest_root'] + params['archer_fielddefinitionapppath'] + iModuleID;

		var headers = {	'Authorization' : 'Archer session-id="' + sSessionToken + '"',
						'Content-type': 'application/json',
						'Accept': 'application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8'};
		
		var postBody = {
			"Value": "?$orderby=Name"
		  };

		LogInfo('URL3: ' + url);
		LogVerbose('BODY3: ' + jsonToString(postBody));
		//LogVerbose('HEAD3: ' + headers);

		webCall(url,'GET',headers,jsonToString(postBody),function(err,h,b)
		{
			if(err)
			{
				LogError("ERR3: " + err);
				//LogVerbose("HEAD3: " + h);
				LogError("BODY3: " + b);
				callBack(true,null);
			}
			else
			{
				LogVerbose("Portfolio App Definition: " + b);
				var resultJSON = JSON.parse(b);

				if(typeof resultJSON != 'undefined' && typeof resultJSON[0].RequestedObject != 'undefined')
				{
					LogVerbose("Archer Returned good app data");

					//Thankfully LevelID is included in every field and since Portolio is not a leveled app, we can just grab it off the first field
					SaaSParams['LevelID'] = resultJSON[0].RequestedObject.LevelId;

					for(var iField in resultJSON)
					{
						var sfieldname = resultJSON[iField].RequestedObject.Name.toLowerCase();
						var salias = resultJSON[iField].RequestedObject.Alias.toLowerCase();
						var sguid = resultJSON[iField].RequestedObject.Guid.toLowerCase();

						//LogVerbose("*Looking for: " + sfieldname);

						//Check field name, alias, and guid since client might rename one or more items and GUID could theoretically be used in another app
						if (sfieldname == "bitsight company name" || salias == "bitsight_company_name" || sguid == "d616bf90-7cb7-4858-8f26-fddf66d618c8")
						{
							LogVerbose("Got CompanyName");
							SaaSParams['CompanyName'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "bitsight guid" || salias == "bitsight_guid" || sguid == "ffb06a0b-a5bd-4fff-9250-76f6a757d926")
						{
							SaaSParams['GUID'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "change subscription type" || salias == "change_subscription_type" || sguid == "52465d77-89f4-449b-b38b-a9f5bcf854a1")
						{
							SaaSParams['changesubscriptionType'] = resultJSON[iField].RequestedObject.Id;
							
							ArcherValuesLists[ArcherValuesLists.length]={
									'FieldName':'changesubscriptionType',
									'ValuesListID': resultJSON[iField].RequestedObject.RelatedValuesListId,
									'Prefix':'changesubscriptionType_'
							};
						}
						else if (sfieldname == "company id" || salias == "company_id" || sguid == "5b4118e8-e5e6-4103-8511-f87dd7625a87")
						{
							SaaSParams['company_id'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "current subscription type" || salias == "current_subscription_type" || sguid == "b28ff0da-a9ec-4539-a2f1-7bdcd77a774c")
						{
							SaaSParams['subscripion_type'] = resultJSON[iField].RequestedObject.Id;
							ArcherValuesLists[ArcherValuesLists.length]={
								'FieldName':'subscripion_type',
								'ValuesListID': resultJSON[iField].RequestedObject.RelatedValuesListId,
								'Prefix':'subscriptionType_'
							};
						}
						else if (sfieldname == "description" || salias == "description" || sguid == "2dad4202-95f7-424f-bd9c-b2e2dd15ea32")
						{
							SaaSParams['description'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "domain" || salias == "domain" || sguid == "a3a80d7d-8fa9-4648-86e7-288380219ce0")
						{
							SaaSParams['domain'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "headline ratings expiration date" || salias == "headline_ratings_expiration_date" || sguid == "f54966ac-1337-402e-8fb2-cfa0f6fe49b6")
						{
							SaaSParams['headlineratingexpirationdate'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "industry" || salias == "industry" || sguid == "425f8d2d-fd24-4013-b9e2-ca904514c63f")
						{
							SaaSParams['industry'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "last risk vectors update" || salias == "last_total_risk_monitoring_update" || sguid == "62e6d1ba-8181-45f7-abb2-6adec774ecf2")
						{
							SaaSParams['last_total_risk_monitoring_update'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "license bitsight headline ratings?" || salias == "license_1_year_of_bitsight_ratings" || sguid == "18b5a260-2447-4b10-a5b6-22980b5f4415")
						{
							SaaSParams['licence1yearofratings'] = resultJSON[iField].RequestedObject.Id;
							ArcherValuesLists[ArcherValuesLists.length]={
								'FieldName':'licence1yearofratings',
								'ValuesListID': resultJSON[iField].RequestedObject.RelatedValuesListId,
								'Prefix':'licence1yearofratings_'
							};
						}
						else if (sfieldname == "pull today's bitsight risk vectors?" || salias == "pull_todays_bitsight_risk_vectors" || sguid == "ce47490e-aaca-4aed-a50f-332ec4ff210f")
						{
							SaaSParams['pulltodaysbitsightriskvectors'] = resultJSON[iField].RequestedObject.Id;
							ArcherValuesLists[ArcherValuesLists.length]={
								'FieldName':'pulltodaysbitsightriskvectors',
								'ValuesListID': resultJSON[iField].RequestedObject.RelatedValuesListId,
								'Prefix':'pulltodaysbitsightriskvectors_'
							};
						}
						else if (sfieldname == "latest rating" || salias == "rating" || sguid == "e6c8ae53-3e49-40f3-b201-3ec18ed661c1")
						{
							SaaSParams['rating'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "rating date" || salias == "rating_date" || sguid == "cd34aff5-3f48-47be-a629-87b2579d38a8")
						{
							SaaSParams['rating_date'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "rating category" || salias == "rating_category" || sguid == "d9fb6ee5-f8fd-448e-8e18-fe52278bbe8a")
						{
							SaaSParams['rating_range'] = resultJSON[iField].RequestedObject.Id;
							ArcherValuesLists[ArcherValuesLists.length]={
								'FieldName':'rating_range',
								'ValuesListID': resultJSON[iField].RequestedObject.RelatedValuesListId,
								'Prefix':'range_'
							};
						}
						else if (sfieldname == "related bitsight alerts" || salias == "related_bitsight_alerts" || sguid == "af683a4c-0e3d-45d8-8b3b-bdd0b6a7a7ba")
						{
							SaaSParams['related_bitsight_alerts'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "related third party profile" || salias == "related_third_party_profile" || sguid == "8958bdaa-d270-4418-a7f0-8846b882f0d3")
						{
							SaaSParams['RelatedTPP'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "subscription change results" || salias == "subscription_change_results" || sguid == "c97fdd37-8dc1-4584-a329-996fc5d1b554")
						{
							SaaSParams['subscriptionchangeresults'] = resultJSON[iField].RequestedObject.Id;
						}

						//Vectors
						else if (sfieldname == "botnet infections" || salias == "botnet_infections" || sguid == "6d0425ec-e509-4022-9a9b-1c3458c2c60e")
						{
							SaaSParams['botnet_infections'] = resultJSON[iField].RequestedObject.Id;
							//Only need to get this once since it's a global values list
							ArcherValuesLists[ArcherValuesLists.length]={
								'FieldName':'rating',
								'ValuesListID': resultJSON[iField].RequestedObject.RelatedValuesListId,
								'Prefix':'rating_'
							};
						}
						else if (sfieldname == "desktop software" || salias == "desktop_software" || sguid == "d96b5db0-adf1-4af2-b59c-e846ecfded67")
						{
							SaaSParams['desktop_software'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "dkim records" || salias == "dkim_records" || sguid == "8916401c-dede-4659-a258-11d88a3a7fb3")
						{
							SaaSParams['dkim'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "dnssec" || salias == "dnssec" || sguid == "b249b1c6-7f4b-48fc-ac41-dc4a66209d7a")
						{
							SaaSParams['dnssec'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "file sharing" || salias == "file_sharing" || sguid == "9bad08b0-4332-42a9-8675-66b20aadc775")
						{
							SaaSParams['file_sharing'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "insecure systems" || salias == "insecure_systems" || sguid == "ddd12adb-349f-4493-b628-ab79d1d86923")
						{
							SaaSParams['insecure_systems'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "malware servers" || salias == "malware_servers" || sguid == "e3f7356f-5478-4805-a7b7-4ef863b01179")
						{
							SaaSParams['malware_servers'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "mobile application security" || salias == "mobile_application_security" || sguid == "69bed8f1-d68b-4571-a3b0-04e7409ebdfc")
						{
							SaaSParams['mobile_application_security'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "mobile software" || salias == "mobile_software" || sguid == "f7b8ccb4-80bd-43b6-bf7f-e43f531d4539")
						{
							SaaSParams['mobile_software'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "open ports" || salias == "open_ports" || sguid == "080c4301-0219-42d3-b2c4-6783aa11dc1a")
						{
							SaaSParams['open_ports'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "patching cadence" || salias == "patching_cadence" || sguid == "2d8cbd62-f04f-4b72-b1ec-b41c9ac9bbb7")
						{
							SaaSParams['patching_cadence'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "potentially exploited" || salias == "potentially_exploited" || sguid == "2c0e1cf5-bd7c-4859-ba54-674df3c76978")
						{
							SaaSParams['potentially_exploited'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "security incidents/breaches" || salias == "security_incidents_breaches" || sguid == "63beaccb-1ef6-40f3-a998-3a5d469095a4")
						{
							SaaSParams['data_breaches'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "server software" || salias == "server_software" || sguid == "eb0f9046-2a57-4f82-b0e7-936c242b5c63")
						{
							SaaSParams['server_software'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "spam propagation" || salias == "spam_propagation" || sguid == "ae1805d4-2405-4310-8d99-4fddbf624fd8")
						{
							SaaSParams['spam_propagation'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "spf domains" || salias == "spf_domains" || sguid == "e6eb290b-b986-4ada-b3aa-6f1bfcd6e14e")
						{
							SaaSParams['spf'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "tls/ssl certificates" || salias == "tls_ssl_certificates" || sguid == "e37379d7-c383-4efc-8be5-75c0b352073d")
						{
							SaaSParams['ssl_certificates'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "tls/ssl configurations" || salias == "tls_ssl_configurations" || sguid == "c61e3eec-7ab9-43d0-a1a5-e08b65c1f680")
						{
							SaaSParams['ssl_configurations'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "unsolicited communications" || salias == "unsolicited_communications" || sguid == "395a5113-fab9-4c71-bb12-7b1d60c33dde")
						{
							SaaSParams['unsolicited_comm'] = resultJSON[iField].RequestedObject.Id;
						}
						else if (sfieldname == "web application headers" || salias == "web_application_headers" || sguid == "99ac94c3-bcdc-4521-ae0c-260d773a1e91")
						{
							SaaSParams['application_security'] = resultJSON[iField].RequestedObject.Id; //Web Application Headers
						}
					}

					if (bVerboseLogging === true)
					{
						var fs = require('fs'); 		
						fs.writeFileSync("logs\\" + appPrefix + "\\" + "aParameters1.json", JSON.stringify(SaaSParams));
						LogInfo("Saved to logs\\" + appPrefix + "\\" + "aParameters1.json file");
					}
			
					LogInfo("SaaSParams: " + jsonToString(SaaSParams));
					
					getValuesListValues(0,callBack);				
				}
				else
				{
					LogSaaSError();
					LogError("ERROR Obtaining Portfolio field definition: " + b);
				}
			}
		});
	}

	var getValuesListValues = function(currentValuesList,callBack)
	{
		LogInfo('----------------------------getValuesListValues----------------------------');
		LogInfo('Getting VL: ' + ArcherValuesLists[currentValuesList].FieldName);
		var valueslistID = ArcherValuesLists[currentValuesList].ValuesListID;
		var url =  params['archer_webroot'] + params['archer_rest_root'] + params['archer_valueslistvaluepath'] + valueslistID;

		var headers = {	'Authorization' : 'Archer session-id="' + sSessionToken + '"',
						'Content-type': 'application/json',
						'Accept': 'application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8'};
		
		var postBody = {};

		LogInfo('URL4: ' + url);

		webCall(url,'GET',headers,jsonToString(postBody),function(err,h,b)
		{
			if(err)
			{
				LogError("ERR4: " + err);
				//LogError("HEAD4: " + h);
				LogError("BODY4: " + b);				
			}
			else
			{
				LogVerbose("ValuesList Full Definition: " + b);
				var resultJSON = JSON.parse(b);

				if(typeof resultJSON != 'undefined' && typeof resultJSON[0].RequestedObject != 'undefined')
				{
					LogVerbose("Archer Returned good values list data");

					var id = "";
					var name = "";
					var prefix = ArcherValuesLists[currentValuesList].Prefix;

					for(var i in resultJSON)
					{
						id = resultJSON[i].RequestedObject.Id;
						name = resultJSON[i].RequestedObject.Name;

						//Format name of value to remove spaces and /
						name = name.replace(/\//g, '');
						name = name.replace(/ /g, '');
						
						name = prefix + name; //Set the name of the parameter to fieldname_Value
						SaaSParams[name] = id;//Add to our parameters
					}

					//Iterate through if there were multiple
					//Did we hit the max?
					LogInfo("Current=" + currentValuesList + " Max=" + ArcherValuesLists.length);
					if(currentValuesList >= ArcherValuesLists.length-1)
					{
						LogInfo("Hit maxResults of " + ArcherValuesLists.length);				
						if (bVerboseLogging === true)
						{
							var fs = require('fs'); 			
							fs.writeFileSync("logs\\" + appPrefix + "\\" + "aParameters2.json", JSON.stringify(SaaSParams));
							LogInfo("Saved to logs\\" + appPrefix + "\\" + "aParameters2.json file");
						}

						GetArcherBitSightCompaniesNeedingRatings(callBack);
					}
					else
					{
						//Still have more values lists to iterate through...
						currentValuesList++; //Increment before running again
						LogInfo("Iterating through next ValuesList=" + currentValuesList);				
						getValuesListValues(currentValuesList,callBack)
					}

				}
				else
				{
					LogSaaSError();
					LogError("ERROR Obtaining Portfolio field values list ids: " + b);
				}
			}
		});
	}

	//Start of all the functions to get ids
	getPortfolioModuleID(callBack);
}





/************************************************************************************************************************************************************************************************************************************/
/********	GetArcherBitSightCompaniesNeedingRatings
/************************************************************************************************************************************************************************************************************************************/
function GetArcherBitSightCompaniesNeedingRatings(callBack)
{
	/* build url */
	var url =  params['archer_webroot'] + params['archer_ws_root'] + params['archer_searchpath'];

	var headers = {'content-type' : 'text/xml; charset=utf-8',
			'SOAPAction' : 'http://archer-tech.com/webservices/ExecuteSearch'}

	//Building a search criteria instead of using the report. This way we can return thousands of records instead of having to page through 250 records at a time...
	var sSearchCriteria;
	if (!bIsLegacyIntegration)
	{	
		sSearchCriteria =   '<SearchReport><PageSize>10000</PageSize><MaxRecordCount>10000</MaxRecordCount><DisplayFields>' +
							'<DisplayField name="BitSight Company Name">d616bf90-7cb7-4858-8f26-fddf66d618c8</DisplayField>' +
							'<DisplayField name="BitSight GUID">ffb06a0b-a5bd-4fff-9250-76f6a757d926</DisplayField>' +
							'<DisplayField name="Pull Todays BitSight Risk Vectors?">ce47490e-aaca-4aed-a50f-332ec4ff210f</DisplayField>' +
							'</DisplayFields><Criteria><ModuleCriteria><Module name="BitSight Portfolio">21d4cfb5-e4d2-4c0b-a34f-559e3a0ef004</Module>' +
							'<SortFields /></ModuleCriteria><Filter><Conditions><ValueListFilterCondition name="Value List 1"><Field name="Pull Todays BitSight Risk Vectors?">' + SaaSParams['pulltodaysbitsightriskvectors'] + '</Field>' +
							'<Operator>Contains</Operator><IsNoSelectionIncluded>False</IsNoSelectionIncluded><Values><Value name="Yes">' + SaaSParams['pulltodaysbitsightriskvectors_Yes'] + '</Value>' +
							'</Values></ValueListFilterCondition></Conditions></Filter></Criteria></SearchReport>';
	}
	else
	{
		sSearchCriteria =   '<SearchReport><PageSize>10000</PageSize><MaxRecordCount>10000</MaxRecordCount><DisplayFields>' +
							'<DisplayField name="BitSight Company Name">C9ABC083-CC49-498D-8ACA-FD1BEE4C9A5A</DisplayField>' +
							'<DisplayField name="BitSight GUID">6E965C7C-84A7-4A98-9701-2EF0E980D967</DisplayField>' +
							'<DisplayField name="Pull Todays BitSight Risk Vectors?">4BCA1C22-D7C6-448E-B4A4-F256D40083B1</DisplayField>' +
							'</DisplayFields><Criteria><ModuleCriteria><Module name="BitSight Portfolio">54FE69B0-3C28-4D94-852D-841151634F22</Module>' +
							'<SortFields /></ModuleCriteria><Filter><Conditions><ValueListFilterCondition name="Value List 1"><Field name="Pull Todays BitSight Risk Vectors?">' + SaaSParams['pulltodaysbitsightriskvectors'] + '</Field>' +
							'<Operator>Contains</Operator><IsNoSelectionIncluded>False</IsNoSelectionIncluded><Values><Value name="Yes">' + SaaSParams['pulltodaysbitsightriskvectors_Yes'] + '</Value>' +
							'</Values></ValueListFilterCondition></Conditions></Filter></Criteria></SearchReport>';

	}
	
	LogVerbose('sSearchCriteria: ' + sSearchCriteria);
	
	//Must escape the XML to next inside of the soap request... 
	sSearchCriteria = sSearchCriteria.replace(/&/g, '&amp;')
	               .replace(/</g, '&lt;')
	               .replace(/>/g, '&gt;')
	               .replace(/"/g, '&quot;')
	               .replace(/'/g, '&apos;');
	
	var postBody = "<?xml version=\"1.0\" encoding=\"utf-8\"?><soap:Envelope xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\">" +
				"<soap:Body><ExecuteSearch xmlns=\"http://archer-tech.com/webservices/\"><sessionToken>" + sSessionToken + "</sessionToken>" +
				"<searchOptions>" + sSearchCriteria + "</searchOptions><pageNumber>1</pageNumber></ExecuteSearch></soap:Body></soap:Envelope>";
		
	var results = []; 

	var archerAPI_GetArcherBitSightCompaniesNeedingRatings = function(url){
		LogInfo('----------------------------GetArcherBitSightCompaniesNeedingRatings----------------------------');
		LogInfo('URL4: ' + url);
		LogVerbose('BODY4: ' + postBody);
		//LogVerbose('HEAD4: ' + headers);

		webCall(url,'POST',headers,postBody,function(err,h,b)
		{
			if(err)
			{
				LogError("ERR4: " + err);
				//LogError("HERE HEAD4: " + h);
				LogError("BODY4: " + b);
				callBack(true,null);
			}
			else
			{
				//LogInfo("Raw Report Results: " + b);
				results = b; //This is the result from the webCall
				reviewResults();
			}
		});
	}

	var reviewResults = function()
	{
		//Convert XML to an XMLDOM
		var doc = xmlStringToXmlDoc(results);

		//Check to see if nothing was returned from the search query
		if (typeof doc.getElementsByTagName("ExecuteSearchResult") == "undefined" || 
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0] == "undefined" || 
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0] == "undefined" ||
			typeof doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue == "undefined")
		{
			//There weren't any records
			LogInfo("No new companies needing ratings. Exiting. 01");
			LogSaaSSuccess();
			LogInfo("SUCCESS: Overall Success");
		}
		else  //Need to proceed and check the count anyway.
		{
			//Need to get the xml inside the SOAP request and url decode the results
			var sXML = doc.getElementsByTagName("ExecuteSearchResult")[0].childNodes[0].nodeValue;

			//variable for our json object
			var resultJSON = null;
			//turn the xml results into the json object
			xml2js.parseString(sXML, function (err, result) 
			{
				resultJSON = result; //get the result into the object we can use below;
			});
			
			if (bVerboseLogging === true)
			{
				var fs = require('fs'); 			
				fs.writeFileSync("logs\\" + appPrefix + "\\" + "archerBitSightCompaniesNewRatings.json", JSON.stringify(resultJSON));
				LogInfo("Saved to logs\\" + appPrefix + "\\" + "archerBitSightCompaniesNewRatings.json file");
			}
			
			var iNumCompanies = resultJSON.Records.$.count; //Get the number of record returned
			LogInfo("iNumCompanies=" + iNumCompanies);

			//Check to see if we have any existing records
			if (iNumCompanies == 0)
			{
				LogInfo("No new companies needing ratings. Exiting. 02");
				LogSaaSSuccess();
				LogInfo("SUCCESS: Overall Success");
			}
			else
			{
				compileResults(resultJSON)
			}
		}
	}

	//Results were found, now get the details
	var compileResults = function(resultJSON)
	{
		var iID_BitSightCompanyGUID;
		var iID_BitSightCompanyName;
		var iID_LicenseBitSightRatings;


		//Get the LevelID of the Portfolio app
		var sArcherBitSightPortfolio_LevelID = resultJSON.Records.LevelCounts[0].LevelCount[0].$.id;

		//Iterate through the FieldDefinition to get the field ids that we care about
		for (var h in resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition) 
		{
			var sAlias = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.alias;
			var sID = resultJSON.Records.Metadata[0].FieldDefinitions[0].FieldDefinition[h].$.id;
				 		
			//LogInfo("  Alias:" + sAlias + "   ID: " + sID);
			if (sAlias == "BitSight_Company_Name"){iID_BitSightCompanyName = sID;}
			else if (sAlias == "BitSight_GUID"){iID_BitSightCompanyGUID = sID;}
			else if (sAlias == "Pull_Todays_BitSight_Risk_Vectors"){iID_LicenseBitSightRatings = sID;}
		}
		
		var sContentID = "";
		var sBitSightCompanyName;
		var sBitSightCompanyGUID;
		var sLicenseBitSightRatings; //Customer driven ID to help with matching


        //Iterate through each Archer company record....
		for (var i in resultJSON.Records.Record)
		{
			LogInfo("-----ARCHER COMPANY RECORD #" + i + "-------");
			sContentID = null;
			sBitSightCompanyName= "";
			sBitSightCompanyGUID= "";
			sLicenseBitSightRatings= ""; //Customer driven ID to help with matching

			
			//Get the Tracking ID...
			sContentID = resultJSON.Records.Record[i].$.contentId;
			//LogInfo("sContentID: " + sContentID);

			//Iterate through the Field elements for the current config record to get the goodies
			for (var y in resultJSON.Records.Record[i].Field) 
			{
				//Get the id of the field because we need to match on the ones we care about...
				sID = resultJSON.Records.Record[i].Field[y].$.id;
				
				//Now find all the good data we care about...
				if (sID == iID_BitSightCompanyName)
				{
					sBitSightCompanyName =  GetArcherText(resultJSON.Records.Record[i].Field[y]);
				}
				else if (sID == iID_BitSightCompanyGUID) 
				{
					sBitSightCompanyGUID = GetArcherText(resultJSON.Records.Record[i].Field[y]);
				}
				else if (sID == iID_LicenseBitSightRatings) 
				{
                    sLicenseBitSightRatings = GetArcherValue(resultJSON.Records.Record[i].Field[y]);
				}
			}
			
			LogInfo("Content ID: " + sContentID + " CompanyName:" + sBitSightCompanyName);
			//Populate the main record with the details we care about.... 
			aBitSightRatings[aBitSightRatings.length] = {
				"ContentID":sContentID,
				"CompanyName":sBitSightCompanyName,
				"guid":sBitSightCompanyGUID,
				"LicenseBitSightRatings":sLicenseBitSightRatings,
			};

			LogInfo("*Company Number#" + aBitSightRatings.length + "=" + JSON.stringify(aBitSightRatings[aBitSightRatings.length-1]));			
		}
       
		if (bVerboseLogging === true)
		{
			var fs = require('fs'); 
            fs.writeFileSync("logs\\" + appPrefix + "\\" + "archerBitSightCompaniesNewRatingsParsed.json", JSON.stringify(aBitSightRatings));
			LogInfo("Saved to logs\\" + appPrefix + "\\" + "archerBitSightCompaniesNewRatingsParsed.json file");
		}

		//Now that we have the current info, now go get the info from the BitSight API
		GetCurrentBitSightCompanyInfo(callBack);
	}

	/* kick off process */
	archerAPI_GetArcherBitSightCompaniesNeedingRatings(url);
}



/************************************************************************************************************************************************************************************************************************************/
/********	GetCurrentBitSightCompanyInfo
/************************************************************************************************************************************************************************************************************************************/
function GetCurrentBitSightCompanyInfo(callBack)
{
	var aBitSightCompanyInfo;
	var bitSightAPI_GetCurrentBitSightCompanyInfo = function(callBack)
	{
        var url = params['bitsight_webroot'] + params['bitsight_currentuser'];

        var sAuthorization = makeBasicAuth(params['bitsight_token']);

        var headers = { 'Accept' : 'application/json',
                        'Authorization' : 'Basic ' + sAuthorization
					};

        var postBody = null;
		
		LogInfo("----------------------GetCurrentBitSightCompanyInfo----------------------");
		LogInfo('URL5: ' + url);
		LogVerbose('BODY5: ' + JSON.stringify(postBody));
		LogInfo('HEAD5: ' + JSON.stringify(headers));

		/* build options */
		var options = {
			method: 'GET',
			uri: url,
			headers: headers,
			body: postBody,
			timeout: 30000,
			rejectUnauthorized: verifyCerts
		}
		
		/* make the request */
		request(options,function (err1, response1, body)
		{
			var errMessage = null;
			
			/* check for error */
			if(err1)
			{

				errMessage='HTTP ERROR: ' + err1;
			}
			else
			{
				//LogInfo("Header Response: " + response1.headers);
				if(response1.statusCode!=200)
				{
					errMessage='HTTP ' + response1.statusCode + ' ERROR: ' + err1;
				}
			}
			/* check for error */
			if(errMessage)
			{
				LogError("ERR5: " + errMessage);
				LogError("BODY5: " + body);
				captureError('Error in HTTP Response: ' + errMessage);
				callBack(true,null);
			}
			else
			{
                LogInfo("body:" + body);
				aBitSightCompanyInfo = JSON.parse(body);
                LogInfo("----------------------GOT Company Info----------------------");
                
                //Just for testing...save to file...
                if (bVerboseLogging === true)
                {
                    var fs = require('fs'); 
                    fs.writeFileSync("logs\\" + appPrefix + "\\" + "aBitSightCompanyInfo.json", JSON.stringify(aBitSightCompanyInfo));
                    LogInfo("Saved to logs\\" + appPrefix + "\\" + "aBitSightCompanyInfo.json file");
                }

				if (typeof aBitSightCompanyInfo == 'undefined' || 
					typeof aBitSightCompanyInfo.customer == 'undefined' || 
					typeof aBitSightCompanyInfo.customer.name == 'undefined')
				{
					BitSightCustomerName = "unknown";
				}else{					
					BitSightCustomerName = aBitSightCompanyInfo.customer.name;
					LogInfo("BitSight Customer: " + BitSightCustomerName);
				}

				GetRatings(callBack);
			}
		});
	}

	/* kick off process */
    bitSightAPI_GetCurrentBitSightCompanyInfo(callBack);
}


/************************************************************************************************************************************************************************************************************************************/
/********	GetRatings
/************************************************************************************************************************************************************************************************************************************/
function GetRatings(callBack)
{
	var guid = aBitSightRatings[iCurrentCompany].guid;
	LogInfo('----------------------------GetRatings----------------------------');
	LogInfo("Company: " + aBitSightRatings[iCurrentCompany].CompanyName + "  GUID: " + guid);
	var url = params['bitsight_webroot'] + params['bitsight_getbaaratings'] + guid + '/assessments';

    var sAuthorization = makeBasicAuth(params['bitsight_token']);

    var headers = { 'Accept' : 'application/json',
                    'Authorization' : 'Basic ' + sAuthorization,
					'Content-type': 'application/json',
					'X-BITSIGHT-CONNECTOR-NAME-VERSION':COST_ArcherBitSightVersion,
					'X-BITSIGHT-CALLING-PLATFORM-VERSION':COST_Platform,
					'X-BITSIGHT-CUSTOMER':BitSightCustomerName
                    };

	var assessmentGUID = params['bitsight_baa_assessment'];
	var postBody =  {
					"assessment_guids": [
						assessmentGUID
					],
					"use_license": "True"
				};
	
	LogInfo('URL6: ' + url);
	LogVerbose('BODY6: ' + JSON.stringify(postBody));

	/* build options */
	var options = {
		method: 'POST',
		uri: url,
		headers: headers,
		body: JSON.stringify(postBody),
		timeout: 30000,
		rejectUnauthorized: verifyCerts
	}
	
	var ratingsResponse;

	/* make the request */
	request(options,function (err1, response, body)
	{
		var errMessage = null;
		//var bUnsubcribed = false;
		
		/* check for error */
		if(err1)
		{
			errMessage='HTTP ERROR: ' + err1;
		}
		else
		{
			//LogInfo("Header Response: " + response.headers);
			// if(response.statusCode==403)
			// {
			// 	//Error 403 is when the subscription is no longer available. Need to identifiy this and set the status in the return results.
			// 	bUnsubcribed = true;
			// }
			//else if(response.statusCode!=200)
			if(response.statusCode!=200)
			{
				errMessage='HTTP ' + response.statusCode + ' ERROR: ' + err1;
			}
		}
		/* check for error */
		if(errMessage)
		{
			LogError("ERR6: " + errMessage);
			LogError("BODY6: " + body);
			captureError('Error in HTTP Response: ' + errMessage);
			callBack(true,null);
		}
		else
		{
			LogInfo("----------------------GOT GetRatings----------------------");
			LogInfo("Raw Results body: " + body);
			ratingsResponse = JSON.parse(body);
		
			if (typeof ratingsResponse == "undefined" || 
				typeof ratingsResponse.risk_vector_grades == "undefined" || 
				typeof ratingsResponse.risk_vector_grades[0] == "undefined")
			{
				//Nothing returned for this company?
				LogWarn("JSON Body is undefined.");
				LogWarn("No results from BitSight API for this company: " + aBitSightRatings[iCurrentCompany].CompanyName);
			}
			else
			{
				if (typeof ratingsResponse.rating_as_of != "undefined")				
				{
					aBitSightRatings[iCurrentCompany].lastriskvectorsupdate = ratingsResponse.rating_as_of;
				}

				//Iterate through and get the vector information
				for (var v in ratingsResponse.risk_vector_grades)
				{
					var slug;
					var grade;

					if (typeof ratingsResponse.risk_vector_grades[v].slug == 'undefined'){
						slug = "";
					}else{
						slug = ratingsResponse.risk_vector_grades[v].slug;
					}

					if (typeof ratingsResponse.risk_vector_grades[v].grade == 'undefined'){
						grade = "N/A";
					}else{
						grade = ratingsResponse.risk_vector_grades[v].grade;
					}

					if (slug == "botnet_infections")
					{
						aBitSightRatings[iCurrentCompany].botnet_infections = grade;
					}
					else if (slug == "spam_propagation")
					{
						aBitSightRatings[iCurrentCompany].spam_propagation = grade;
					}
					else if (slug == "malware_servers")
					{
						aBitSightRatings[iCurrentCompany].malware_servers = grade;
					}
					else if (slug == "unsolicited_comm")
					{
						aBitSightRatings[iCurrentCompany].unsolicited_comm = grade;
					}
					else if (slug == "potentially_exploited")
					{
						aBitSightRatings[iCurrentCompany].potentially_exploited = grade;
					}
					else if (slug == "spf")
					{
						aBitSightRatings[iCurrentCompany].spf = grade;
					}
					else if (slug == "dkim")
					{
						aBitSightRatings[iCurrentCompany].dkim = grade;
					}
					else if (slug == "ssl_certificates")
					{
						aBitSightRatings[iCurrentCompany].ssl_certificates = grade;
					}
					else if (slug == "ssl_configurations")
					{
						aBitSightRatings[iCurrentCompany].ssl_configurations = grade;
					}
					else if (slug == "open_ports")
					{
						aBitSightRatings[iCurrentCompany].open_ports = grade;
					}
					else if (slug == "application_security")
					{
						aBitSightRatings[iCurrentCompany].application_security = grade;
					}
					else if (slug == "patching_cadence")
					{
						aBitSightRatings[iCurrentCompany].patching_cadence = grade;
					}
					else if (slug == "insecure_systems")
					{
						aBitSightRatings[iCurrentCompany].insecure_systems = grade;
					}
					else if (slug == "server_software")
					{
						aBitSightRatings[iCurrentCompany].server_software = grade;
					}
					else if (slug == "desktop_software")
					{
						aBitSightRatings[iCurrentCompany].desktop_software = grade;
					}
					else if (slug == "mobile_software")
					{
						aBitSightRatings[iCurrentCompany].mobile_software = grade;
					}
					else if (slug == "dnssec")
					{
						aBitSightRatings[iCurrentCompany].dnssec = grade;
					}
					else if (slug == "mobile_application_security")
					{
						aBitSightRatings[iCurrentCompany].mobile_application_security = grade;
					}
					else if (slug == "file_sharing")
					{
						aBitSightRatings[iCurrentCompany].file_sharing = grade;
					}
					else if (slug == "data_breaches")
					{
						aBitSightRatings[iCurrentCompany].data_breaches = grade;
					}
				}
			}


			//Iterate through if there were multiple companies
			//Did we hit the max?
			LogInfo("Current=" + iCurrentCompany + " Max=" + aBitSightRatings.length);
			if(iCurrentCompany >= aBitSightRatings.length-1)
			{
				LogInfo("Hit maxResults of " + aBitSightRatings.length);				
				//Just for testing...save to file...
				if (bVerboseLogging === true)
				{
					var fs = require('fs'); 			
					fs.writeFileSync("logs\\" + appPrefix + "\\" + "aBitSightRatings.json", JSON.stringify(aBitSightRatings));
					LogInfo("Saved to logs\\" + appPrefix + "\\" + "aBitSightRatings.json file");
				}

    			DetermineExportPath(callBack);
			}
			else
			{
				//Still have more companies to iterate through...
				iCurrentCompany++; //Increment before running again
				LogInfo("Iterating through next company=" + iCurrentCompany);				
				GetRatings(callBack)
			}
		}
	});
}

function DetermineExportPath(callBack)
{
    if (params['bIsSaaS'] == "true")
    {
        //This functionality is only for SaaS/Hosted clients running this script outside of Archer (Node.js Prompt).
        if (aBitSightRatings.length == 0)
        {
            LogInfo('Nothing to send back to Archer...exiting.');
            LogSaaSSuccess();
        }
        else
        {
            formatDataForSaaSUpload(callBack);
        }
    }
    else
    {
        //Return data back to Archer data feed
        LogInfo('Non-SaaS - returning data to Archer.');	
        ReturnDataToArcher(callBack);
    }
}



/********************************************************/
/********	ReturnDataToArcher
/********************************************************/
var ReturnDataToArcher = function(callBack)
{
	LogInfo('--------------Non-SaaS - ReturnDataToArcher-------------------------------');
	LogInfo('--------------Generating XML for Archer Records---------------------------');
	var start = Buffer.from('<Records>\n', 'utf8');
	var xmlData = jsonArrayToXMLBuffer(aBitSightRatings,'Record');
	var end = Buffer.from('</Records>\n', 'utf8');
	var returntoArcher = Buffer.concat([start,xmlData,end]);

	var sXMLReturn = returntoArcher.toString('utf8');
	//LogInfo("XML: " + sXMLReturn);
	
    LogInfo("Finish with callback...");
	callBack(false,sXMLReturn);
}



/********************************************************/
/********	formatDataForSaaSUpload
/********************************************************/
var formatDataForSaaSUpload = function(callBack)
{

	LogInfo('----------------------formatDataForSaaSUpload START------------------------------');

	var postBody = [];  //Array formatted as the postBody that will loop through to create records later

    var LevelID = SaaSParams['LevelID'];
    var lastriskvectorsupdate = SaaSParams['last_total_risk_monitoring_update'];
	var pulltodaysbitsightriskvectors = SaaSParams['pulltodaysbitsightriskvectors'];
	var pulltodaysbitsightriskvectors_No = SaaSParams['pulltodaysbitsightriskvectors_No'];	

    var botnet_infections = SaaSParams['botnet_infections'];
    var spam_propagation = SaaSParams['spam_propagation'];
    var malware_servers = SaaSParams['malware_servers'];
    var unsolicited_comm = SaaSParams['unsolicited_comm'];
    var potentially_exploited = SaaSParams['potentially_exploited'];
    var spf = SaaSParams['spf'];
    var dkim = SaaSParams['dkim'];
    var ssl_certificates = SaaSParams['ssl_certificates'];
    var ssl_configurations = SaaSParams['ssl_configurations'];
    var open_ports = SaaSParams['open_ports'];
    var application_security = SaaSParams['application_security'];
    var patching_cadence = SaaSParams['patching_cadence'];
    var insecure_systems = SaaSParams['insecure_systems'];
    var server_software = SaaSParams['server_software'];
    var desktop_software = SaaSParams['desktop_software'];
    var mobile_software = SaaSParams['mobile_software'];
    var dnssec = SaaSParams['dnssec'];
    var mobile_application_security = SaaSParams['mobile_application_security'];
    var file_sharing = SaaSParams['file_sharing'];
    var data_breaches = SaaSParams['data_breaches'];	

	LogInfo("Count of aBitSightRatings: " + aBitSightRatings.length);
	
	if (aBitSightRatings.length == 0)
	{
		LogInfo('Nothing to send back to Archer...exiting.');
		LogSaaSSuccess();
	}
	else
	{
		for(var i in aBitSightRatings)
		{
			LogInfo("  - Company = " + i);

			var contentID = parseInt(aBitSightRatings[i].ContentID);

			var botnet_infections_vl = GetRatingListValue(aBitSightRatings[i].botnet_infections);
            var spam_propagation_vl = GetRatingListValue(aBitSightRatings[i].spam_propagation);
            var malware_servers_vl = GetRatingListValue(aBitSightRatings[i].malware_servers);
            var unsolicited_comm_vl = GetRatingListValue(aBitSightRatings[i].unsolicited_comm);
            var potentially_exploited_vl = GetRatingListValue(aBitSightRatings[i].potentially_exploited);
            var spf_vl = GetRatingListValue(aBitSightRatings[i].spf);
            var dkim_vl = GetRatingListValue(aBitSightRatings[i].dkim);
            var ssl_certificates_vl = GetRatingListValue(aBitSightRatings[i].ssl_certificates);
            var ssl_configurations_vl = GetRatingListValue(aBitSightRatings[i].ssl_configurations);
            var open_ports_vl = GetRatingListValue(aBitSightRatings[i].open_ports);
            var application_security_vl = GetRatingListValue(aBitSightRatings[i].application_security);
            var patching_cadence_vl = GetRatingListValue(aBitSightRatings[i].patching_cadence);
            var insecure_systems_vl = GetRatingListValue(aBitSightRatings[i].insecure_systems);
            var server_software_vl = GetRatingListValue(aBitSightRatings[i].server_software);
            var desktop_software_vl = GetRatingListValue(aBitSightRatings[i].desktop_software);
            var mobile_software_vl = GetRatingListValue(aBitSightRatings[i].mobile_software);
            var dnssec_vl = GetRatingListValue(aBitSightRatings[i].dnssec);
            var mobile_application_security_vl = GetRatingListValue(aBitSightRatings[i].mobile_application_security);
            var file_sharing_vl = GetRatingListValue(aBitSightRatings[i].file_sharing);
            var data_breaches_vl = GetRatingListValue(aBitSightRatings[i].data_breaches);

			postBody[i] = {
				"Content": {
					"LevelId": LevelID,
					"Tag": "BitSightPortfolio",
					"Id": contentID,
					"FieldContents": {
						[lastriskvectorsupdate]:
						{
							"Type": ArcherInputTypes['date'],
							"Tag": "lastriskvectorsupdate",
							"Value": aBitSightRatings[i].lastriskvectorsupdate,
							"FieldID": lastriskvectorsupdate
						},
						[pulltodaysbitsightriskvectors]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "pulltodaysbitsightriskvectors",
							"Value": {
								"ValuesListIds": [
									pulltodaysbitsightriskvectors_No
								]
								},
							"FieldID": pulltodaysbitsightriskvectors
						},					
						[botnet_infections]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "botnet_infections",
							"Value": {
								"ValuesListIds": [
									botnet_infections_vl
								]
							},
							"FieldID": botnet_infections
						},
						[spam_propagation]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "spam_propagation",
							"Value": {
								"ValuesListIds": [
									spam_propagation_vl
								]
							},
							"FieldID": spam_propagation
						},
						[malware_servers]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "malware_servers",
							"Value": {
								"ValuesListIds": [
									malware_servers_vl
								]
							},
							"FieldID": malware_servers
						},
						[unsolicited_comm]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "unsolicited_comm",
							"Value": {
								"ValuesListIds": [
									unsolicited_comm_vl
								]
							},
							"FieldID": unsolicited_comm
						},
						[potentially_exploited]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "potentially_exploited",
							"Value": {
								"ValuesListIds": [
									potentially_exploited_vl
								]
							},
							"FieldID": potentially_exploited
						},
						[spf]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "spf",
							"Value": {
								"ValuesListIds": [
									spf_vl
								]
							},
							"FieldID": spf
						},
						[dkim]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "dkim",
							"Value": {
								"ValuesListIds": [
									dkim_vl
								]
							},
							"FieldID": dkim
						},
						[ssl_certificates]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "ssl_certificates",
							"Value": {
								"ValuesListIds": [
									ssl_certificates_vl
								]
							},
							"FieldID": ssl_certificates
						},
						[ssl_configurations]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "ssl_configurations",
							"Value": {
								"ValuesListIds": [
									ssl_configurations_vl
								]
							},
							"FieldID": ssl_configurations
						},
						[open_ports]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "open_ports",
							"Value": {
								"ValuesListIds": [
									open_ports_vl
								]
							},
							"FieldID": open_ports
						},
						[application_security]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "application_security",
							"Value": {
								"ValuesListIds": [
									application_security_vl
								]
							},
							"FieldID": application_security
						},
						[patching_cadence]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "patching_cadence",
							"Value": {
								"ValuesListIds": [
									patching_cadence_vl
								]
							},
							"FieldID": patching_cadence
						},
						[insecure_systems]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "insecure_systems",
							"Value": {
								"ValuesListIds": [
									insecure_systems_vl
								]
							},
							"FieldID": insecure_systems
						},
						[server_software]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "server_software",
							"Value": {
								"ValuesListIds": [
									server_software_vl
								]
							},
							"FieldID": server_software
						},
						[desktop_software]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "desktop_software",
							"Value": {
								"ValuesListIds": [
									desktop_software_vl
								]
							},
							"FieldID": desktop_software
						},
						[mobile_software]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "mobile_software",
							"Value": {
								"ValuesListIds": [
									mobile_software_vl
								]
							},
							"FieldID": mobile_software
						},
						[dnssec]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "dnssec",
							"Value": {
								"ValuesListIds": [
									dnssec_vl
								]
							},
							"FieldID": dnssec
						},
						[mobile_application_security]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "mobile_application_security",
							"Value": {
								"ValuesListIds": [
									mobile_application_security_vl
								]
							},
							"FieldID": mobile_application_security
						},
						[file_sharing]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "file_sharing",
							"Value": {
								"ValuesListIds": [
									file_sharing_vl
								]
							},
							"FieldID": file_sharing
						},
						[data_breaches]:
						{
							"Type": ArcherInputTypes['valueslist'],
							"Tag": "data_breaches",
							"Value": {
								"ValuesListIds": [
									data_breaches_vl
								]
							},
							"FieldID": data_breaches
						}
					}
				}
			}; //End of postBody

			LogVerbose("postBody[" + i + "] = " +  jsonToString(postBody[i]));

		} //End of for loop for iteration

		//reset iCurrentCompany counter for iteration for updating Archere
		iCurrentCompany = 0;
		return SaaSuploadToArcher(postBody);
	
	}//End of if statement when there is something to return to Archer

} //End of formatDataForSaaSUpload function

function GetRatingListValue(sLetter)
{
    if (typeof sLetter == 'undefined' || sLetter == null || sLetter == "" || sLetter == "N/A")
    {
        return SaaSParams['rating_NA'];
    }
    sLetter = sLetter.toLowerCase();
    if (sLetter == 'a')
    {
        return SaaSParams['rating_A'];
    }else if (sLetter == 'b')
    {
        return SaaSParams['rating_B'];
    }else if (sLetter == 'c')
    {
        return SaaSParams['rating_C'];
    }else if (sLetter == 'd')
    {
        return SaaSParams['rating_D'];
    }else if (sLetter == 'f')
    {
        return SaaSParams['rating_F'];
    }else{
        return SaaSParams['rating_NA'];
    }
}


/********************************************************/
/********	SaaSuploadToArcher
/********************************************************/
var SaaSuploadToArcher = function(postBody)
{
	LogInfo("STARTING ON RECORD #: " + iCurrentCompany);

	LogInfo("ContentID: " + postBody[iCurrentCompany].Content.Id);

	var sMETHOD = "";
	sMETHOD = "PUT";
	LogInfo('----------------------------UPDATE RECORD----------------------------');    
	    
	var url =  params['archer_webroot'] + params['archer_rest_root'] + params['archer_contentpath'];

	var headers = {	'Authorization' : 'Archer session-id="' + sSessionToken + '"',
					'Content-type': 'application/json',
					'Accept': 'application/json,text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8'};
    
    LogInfo('URL7: ' + url);
	LogVerbose('BODY7: ' + jsonToString(postBody[iCurrentCompany]));

    webCall(url,sMETHOD,headers,jsonToString(postBody[iCurrentCompany]),function(err,h,b)
    {
        if(err)
        {
            LogError("ERR7: " + err);
            //LogError("HEAD7: " + h);
            LogError("BODY7: " + b);
        }
        else
        {
            //LogInfo("Raw Report Results: " + b);
            var resultJSON = JSON.parse(b);

            //Just for testing...save to file...
            if (bVerboseLogging === true)
            {
                var fs = require('fs'); 			
                fs.writeFileSync("logs\\" + appPrefix + "\\" + "resultJSONUpdateRecord" + iCurrentCompany + ".json", JSON.stringify(resultJSON));
                LogInfo("Saved to logs\\" + appPrefix + "\\" + "resultJSONUpdateRecord" + iCurrentCompany + ".json file");
            }
            
            var iContentID = resultJSON.RequestedObject.Id; //Get the content ID
            LogInfo("iContentID created: " + iContentID);

			//Iterate through if there were multiple records to create
			//Did we hit the max?		
			LogInfo("Current=" + iCurrentCompany + " Max=" + postBody.length);
			if(iCurrentCompany >= postBody.length-1)
			{
				LogInfo("Hit maxResults of " + postBody.length);
				LogSaaSSuccess();
                LogInfo("SUCCESS: SaaS Records Created");
                return true;
			}
			else
			{
				//Still have more records to iterate through...
				LogInfo("Iterating through next record=" + iCurrentCompany);
				iCurrentCompany++; //Increment before running again
				return SaaSuploadToArcher(postBody);
			}
        }
    });
} //End of uploadToArcher function




/************************************************************************************************************************************************************************************************************************************/
/******** DATAFEED BEGINS HERE
/************************************************************************************************************************************************************************************************************************************/
var lrt = init();
ArcherAPI_Login(dfCallback);